// 全局数据存储
let currentData = null;

// 工具函数：从文本中提取UUID
function extractUuidFromText(text) {
    if (!text) return null;
    const uuidMatch = text.match(/offloaded with uuid:\s*([a-f0-9-]{36})/i);
    return uuidMatch ? uuidMatch[1] : null;
}

// 工具函数：从offloadContext获取原始内容
function getOriginalContentFromOffload(offloadContext, uuid) {
    if (!offloadContext || !uuid) return null;
    return offloadContext[uuid] || null;
}

// 工具函数：计算消息文本长度
function calculateTextLength(messages) {
    let totalLength = 0;
    if (!messages || !Array.isArray(messages)) return 0;
    
    messages.forEach(msg => {
        if (msg.content && Array.isArray(msg.content)) {
            msg.content.forEach(item => {
                if (item.type === 'text' && item.text) {
                    totalLength += item.text.length;
                } else if (item.type === 'thinking' && item.thinking) {
                    totalLength += item.thinking.length;
                } else if (item.type === 'tool_use' && item.input) {
                    totalLength += JSON.stringify(item.input).length;
                } else if (item.type === 'tool_result' && item.output) {
                    if (Array.isArray(item.output)) {
                        item.output.forEach(o => {
                            if (o.type === 'text' && o.text) {
                                totalLength += o.text.length;
                            } else {
                                totalLength += JSON.stringify(o).length;
                            }
                        });
                    } else {
                        totalLength += JSON.stringify(item.output).length;
                    }
                }
            });
        }
    });
    return totalLength;
}

// 工具函数：获取消息预览
function getMessagePreview(msg) {
    if (!msg.content || !Array.isArray(msg.content)) {
        return '无内容';
    }
    
    for (let item of msg.content) {
        if (item.type === 'thinking') {
            continue;
        }
        if (item.type === 'text' && item.text) {
            return item.text.substring(0, 100) + (item.text.length > 100 ? '...' : '');
        } else if (item.type === 'tool_use') {
            return `工具调用: ${item.name || 'unknown'}`;
        } else if (item.type === 'tool_result') {
            return `工具结果: ${item.name || 'unknown'}`;
        }
    }
    
    return '查看详情...';
}

// 获取完整的消息预览文本（不截断），与列表页显示格式一致
function getFullMessagePreview(msg) {
    if (!msg.content || !Array.isArray(msg.content)) {
        return '无内容';
    }
    
    // 检查是否是工具消息
    const toolItem = msg.content.find(item => item.type === 'tool_use' || item.type === 'tool_result');
    if (toolItem) {
        if (toolItem.type === 'tool_use') {
            let params = '参数：  ';
            if (toolItem.input && typeof toolItem.input === 'object') {
                const paramList = [];
                for (const [key, value] of Object.entries(toolItem.input)) {
                    const valueStr = typeof value === 'string' ? value : JSON.stringify(value);
                    paramList.push(`${key}:${valueStr}`);
                }
                params += paramList.join('  ');
            } else if (toolItem.input) {
                params += JSON.stringify(toolItem.input);
            } else {
                params += '无参数';
            }
            return params;
        } else if (toolItem.type === 'tool_result') {
            let result = '结果：  ';
            if (toolItem.output) {
                if (Array.isArray(toolItem.output)) {
                    const returns = toolItem.output.map(o => {
                        if (o.type === 'text' && o.text) {
                            return o.text;
                        } else {
                            return JSON.stringify(o);
                        }
                    }).join(' ');
                    result += returns;
                } else {
                    const returnStr = typeof toolItem.output === 'string' 
                        ? toolItem.output 
                        : JSON.stringify(toolItem.output);
                    result += returnStr;
                }
            } else {
                result += '无返回内容';
            }
            return result;
        }
    }
    
    // 普通消息，获取所有文本内容（完整，不截断）
    let fullText = '';
    for (let item of msg.content) {
        if (item.type === 'thinking') {
            continue;
        }
        if (item.type === 'text' && item.text) {
            fullText += item.text;
        }
    }
    
    return fullText || '查看详情...';
}

// 工具函数：创建内容项
function createContentItem(contentItem, index) {
    const div = document.createElement('div');
    div.className = 'content-item';

    const typeLabel = document.createElement('div');
    typeLabel.className = 'content-type';
    
    let contentText = '';
    let itemClass = '';
    let isThinking = false;
    let isCompressed = false;

    if (contentItem.type === 'text') {
        typeLabel.textContent = 'Text';
        contentText = contentItem.text || '';
        // 注意：contentItem本身不包含metadata，压缩判断应该在消息级别进行
        // 这里保留原有逻辑作为备用，但主要判断应该在createMessageElement中
        if (contentText.includes('<compressed') || contentText.includes('compressed_current_round')) {
            isCompressed = true;
            typeLabel.textContent = 'Text (Compressed)';
            typeLabel.style.color = '#856404';
            typeLabel.style.fontWeight = 'bold';
        }
    } else if (contentItem.type === 'tool_use') {
        typeLabel.textContent = 'TOOL USE';
        itemClass = 'tool-use';
        if (contentItem.input && typeof contentItem.input === 'object') {
            const params = [];
            for (const [key, value] of Object.entries(contentItem.input)) {
                const valueStr = typeof value === 'string' ? value : JSON.stringify(value);
                params.push(`${key}:${valueStr}`);
            }
            contentText = `工具调用参数：  ${params.join('  ')}`;
        } else if (contentItem.input) {
            contentText = `工具调用参数：  ${JSON.stringify(contentItem.input)}`;
        } else {
            contentText = '工具调用参数：  无参数';
        }
    } else if (contentItem.type === 'tool_result') {
        typeLabel.textContent = 'TOOL RESULT';
        itemClass = 'tool-result';
        if (contentItem.output) {
            if (Array.isArray(contentItem.output)) {
                const returns = contentItem.output.map(o => {
                    if (o.type === 'text' && o.text) {
                        return o.text;
                    } else {
                        return JSON.stringify(o);
                    }
                }).join(' ');
                contentText = `工具调用返回：  ${returns}`;
            } else {
                const returnStr = typeof contentItem.output === 'string' 
                    ? contentItem.output 
                    : JSON.stringify(contentItem.output);
                contentText = `工具调用返回：  ${returnStr}`;
            }
        } else {
            contentText = '工具调用返回：  无返回内容';
        }
    } else if (contentItem.type === 'thinking') {
        typeLabel.textContent = 'Thinking';
        itemClass = 'thinking';
        contentText = contentItem.thinking || '';
        isThinking = true;
    } else {
        typeLabel.textContent = contentItem.type || 'Unknown';
        contentText = JSON.stringify(contentItem, null, 2);
    }

    div.className = `content-item ${itemClass}`;
    if (isCompressed) {
        div.classList.add('content-compressed');
    }
    
    if (contentItem.type === 'tool_use' || contentItem.type === 'tool_result') {
        typeLabel.style.fontFamily = 'Monaco, "Courier New", monospace';
        typeLabel.style.fontSize = '13px';
        typeLabel.style.fontWeight = 'bold';
        typeLabel.style.marginBottom = '8px';
    }
    
    div.appendChild(typeLabel);

    const textDiv = document.createElement('div');
    if (isThinking) {
        textDiv.className = 'content-text thinking collapsed';
    } else if (contentItem.type === 'tool_use' || contentItem.type === 'tool_result') {
        textDiv.className = 'content-text tool-content';
        textDiv.style.fontFamily = 'Monaco, "Courier New", monospace';
        textDiv.style.fontSize = '12px';
        textDiv.style.background = '#2a2a2a';
        textDiv.style.padding = '8px';
        textDiv.style.borderRadius = '4px';
        textDiv.style.whiteSpace = 'pre-wrap';
        textDiv.style.wordBreak = 'break-word';
        textDiv.style.marginTop = '4px';
        textDiv.style.color = '#e0e0e0';
    } else {
        textDiv.className = 'content-text';
    }
    textDiv.textContent = contentText;
    textDiv.id = `content-${Date.now()}-${index}`;

    if (isThinking) {
        const toggleBtn = document.createElement('div');
        toggleBtn.className = 'thinking-toggle';
        toggleBtn.textContent = '展开思考内容';
        toggleBtn.onclick = function(e) {
            e.stopPropagation();
            if (textDiv.classList.contains('collapsed')) {
                textDiv.classList.remove('collapsed');
                toggleBtn.textContent = '收起思考内容';
            } else {
                textDiv.classList.add('collapsed');
                toggleBtn.textContent = '展开思考内容';
            }
        };
        div.appendChild(textDiv);
        div.appendChild(toggleBtn);
    } else {
        if (contentText.length > 200) {
            const expandBtn = document.createElement('button');
            expandBtn.className = 'expand-btn';
            expandBtn.textContent = '展开';
            expandBtn.onclick = function(e) {
                e.stopPropagation();
                if (textDiv.classList.contains('collapsed')) {
                    textDiv.classList.remove('collapsed');
                    expandBtn.textContent = '收起';
                } else {
                    textDiv.classList.add('collapsed');
                    expandBtn.textContent = '展开';
                }
            };
            textDiv.className = 'content-text collapsed';
            div.appendChild(textDiv);
            div.appendChild(expandBtn);
        } else {
            div.appendChild(textDiv);
        }
    }

    return div;
}

// 工具函数：获取offloadContext
function getOffloadContext(data) {
    if (data.agent && data.agent.memory && data.agent.memory.offloadContext) {
        return data.agent.memory.offloadContext;
    } else if (data.offloadContext) {
        return data.offloadContext;
    }
    return null;
}

// 统计所有消息的token使用量，返回top 10排名Map
// 返回格式: Map<messageId, rank>，rank为1-10，null表示不在top 10
function calculateTokenUsageRanking(data) {
    const tokenUsageMap = new Map(); // messageId -> totalTokens
    const messageIdMap = new Map(); // messageId -> message
    
    // 收集所有消息的token使用量
    function collectTokenUsage(messages) {
        if (!messages || !Array.isArray(messages)) return;
        
        messages.forEach((msg, index) => {
            const chatUsage = (msg.metadata && msg.metadata._chat_usage) || msg._chat_usage;
            if (chatUsage && chatUsage.totalTokens) {
                // 生成唯一的消息标识符
                // 优先使用msg.id，如果没有则使用role+name+timestamp+content的组合
                let messageId = msg.id;
                if (!messageId) {
                    const role = msg.role || 'unknown';
                    const name = msg.name || 'unknown';
                    const timestamp = msg.timestamp || '';
                    // 使用消息内容的前100个字符作为hash（与显示时保持一致）
                    const contentHash = msg.content ? JSON.stringify(msg.content).substring(0, 100) : '';
                    messageId = `${role}_${name}_${timestamp}_${contentHash}`;
                }
                
                const totalTokens = chatUsage.totalTokens || 0;
                
                // 如果同一个消息ID已经存在，取较大的值
                if (tokenUsageMap.has(messageId)) {
                    const existing = tokenUsageMap.get(messageId);
                    if (totalTokens > existing) {
                        tokenUsageMap.set(messageId, totalTokens);
                        messageIdMap.set(messageId, msg);
                    }
                } else {
                    tokenUsageMap.set(messageId, totalTokens);
                    messageIdMap.set(messageId, msg);
                }
            }
        });
    }
    
    // 收集workingMemoryStorage中的消息
    let workingMemory = [];
    if (data.agent && data.agent.memory && data.agent.memory.workingMemoryStorage) {
        workingMemory = data.agent.memory.workingMemoryStorage;
    } else if (data.workingMemoryStorage) {
        workingMemory = data.workingMemoryStorage;
    }
    collectTokenUsage(workingMemory);
    
    // 收集originalMemoryStorage中的消息
    let originalMemory = [];
    if (data.agent && data.agent.memory && data.agent.memory.originalMemoryStorage) {
        originalMemory = data.agent.memory.originalMemoryStorage;
    } else if (data.originalMemoryStorage) {
        originalMemory = data.originalMemoryStorage;
    }
    collectTokenUsage(originalMemory);
    
    // 收集offloadContext中的消息
    const offloadContext = getOffloadContext(data);
    if (offloadContext) {
        Object.values(offloadContext).forEach(messages => {
            if (Array.isArray(messages)) {
                collectTokenUsage(messages);
            }
        });
    }
    
    // 按token使用量排序，找出top 10
    const sortedEntries = Array.from(tokenUsageMap.entries())
        .sort((a, b) => b[1] - a[1]) // 降序排列
        .slice(0, 10); // 取前10个
    
    // 创建排名Map
    const rankingMap = new Map();
    sortedEntries.forEach(([messageId, tokens], index) => {
        rankingMap.set(messageId, index + 1); // 排名从1开始
    });
    
    return rankingMap;
}

// 获取Top N token消耗的消息对象
// 返回格式: Array<{message, totalTokens, rank, messageId}>
function getTopTokenUsageMessages(data, topN = 5) {
    const tokenUsageMap = new Map(); // messageId -> totalTokens
    const messageIdMap = new Map(); // messageId -> message
    
    // 收集所有消息的token使用量（与calculateTokenUsageRanking保持一致的逻辑）
    function collectTokenUsage(messages) {
        if (!messages || !Array.isArray(messages)) return;
        
        messages.forEach((msg, index) => {
            const chatUsage = (msg.metadata && msg.metadata._chat_usage) || msg._chat_usage;
            if (chatUsage && chatUsage.totalTokens) {
                // 生成唯一的消息标识符（与calculateTokenUsageRanking保持一致）
                let messageId = msg.id;
                if (!messageId) {
                    const role = msg.role || 'unknown';
                    const name = msg.name || 'unknown';
                    const timestamp = msg.timestamp || '';
                    // 使用消息内容的前100个字符作为hash（与显示时保持一致）
                    const contentHash = msg.content ? JSON.stringify(msg.content).substring(0, 100) : '';
                    messageId = `${role}_${name}_${timestamp}_${contentHash}`;
                }
                
                const totalTokens = chatUsage.totalTokens || 0;
                
                // 如果同一个消息ID已经存在，取较大的值
                if (tokenUsageMap.has(messageId)) {
                    const existing = tokenUsageMap.get(messageId);
                    if (totalTokens > existing) {
                        tokenUsageMap.set(messageId, totalTokens);
                        messageIdMap.set(messageId, msg);
                    }
                } else {
                    tokenUsageMap.set(messageId, totalTokens);
                    messageIdMap.set(messageId, msg);
                }
            }
        });
    }
    
    // 收集workingMemoryStorage中的消息
    let workingMemory = [];
    if (data.agent && data.agent.memory && data.agent.memory.workingMemoryStorage) {
        workingMemory = data.agent.memory.workingMemoryStorage;
    } else if (data.workingMemoryStorage) {
        workingMemory = data.workingMemoryStorage;
    }
    collectTokenUsage(workingMemory);
    
    // 收集originalMemoryStorage中的消息
    let originalMemory = [];
    if (data.agent && data.agent.memory && data.agent.memory.originalMemoryStorage) {
        originalMemory = data.agent.memory.originalMemoryStorage;
    } else if (data.originalMemoryStorage) {
        originalMemory = data.originalMemoryStorage;
    }
    collectTokenUsage(originalMemory);
    
    // 收集offloadContext中的消息
    const offloadContext = getOffloadContext(data);
    if (offloadContext) {
        Object.values(offloadContext).forEach(messages => {
            if (Array.isArray(messages)) {
                collectTokenUsage(messages);
            }
        });
    }
    
    // 按token使用量排序，找出top N
    const sortedEntries = Array.from(tokenUsageMap.entries())
        .sort((a, b) => b[1] - a[1]) // 降序排列
        .slice(0, topN); // 取前N个
    
    // 返回消息对象数组
    return sortedEntries.map(([messageId, totalTokens], index) => ({
        message: messageIdMap.get(messageId),
        totalTokens: totalTokens,
        rank: index + 1,
        messageId: messageId
    }));
}

