// compression-events.js
// 依赖: common.js

// 存储Token数据，用于窗口大小调整时重绘
let eventsTokenChartData = null;

// 绘制压缩事件Token消耗趋势图表（提前定义，确保在调用时可用）
function drawEventsTokenChart(tokenData, totalTokens) {
    const canvas = document.getElementById('eventsTokenChart');
    if (!canvas) return;

    const ctx = canvas.getContext('2d');
    const width = canvas.offsetWidth || 800;
    const height = 250;
    canvas.width = width;
    canvas.height = height;

    // 清除画布
    ctx.clearRect(0, 0, width, height);

    if (tokenData.length === 0) {
        ctx.fillStyle = '#999';
        ctx.font = '14px Arial';
        ctx.textAlign = 'center';
        ctx.fillText('无Token数据', width / 2, height / 2);
        return;
    }

    // 设置边距
    const padding = { top: 40, right: 40, bottom: 60, left: 60 };
    const chartWidth = width - padding.left - padding.right;
    const chartHeight = height - padding.top - padding.bottom;

    // 计算数据范围
    const maxValue = Math.max(...tokenData.map(d => d.totalTokens), totalTokens / tokenData.length * 2) || 1000;
    const stepX = chartWidth / Math.max(tokenData.length - 1, 1);
    const scaleY = chartHeight / maxValue;

    // 绘制网格线
    ctx.strokeStyle = '#e0e0e0';
    ctx.lineWidth = 1;
    const gridLines = 5;
    for (let i = 0; i <= gridLines; i++) {
        const y = padding.top + (chartHeight / gridLines) * i;
        ctx.beginPath();
        ctx.moveTo(padding.left, y);
        ctx.lineTo(width - padding.right, y);
        ctx.stroke();
    }

    // 绘制坐标轴
    ctx.strokeStyle = '#333';
    ctx.lineWidth = 2;
    // Y轴
    ctx.beginPath();
    ctx.moveTo(padding.left, padding.top);
    ctx.lineTo(padding.left, height - padding.bottom);
    ctx.stroke();
    // X轴
    ctx.beginPath();
    ctx.moveTo(padding.left, height - padding.bottom);
    ctx.lineTo(width - padding.right, height - padding.bottom);
    ctx.stroke();

    // 绘制Y轴标签
    ctx.fillStyle = '#666';
    ctx.font = '12px Arial';
    ctx.textAlign = 'right';
    ctx.textBaseline = 'middle';
    for (let i = 0; i <= gridLines; i++) {
        const value = maxValue - (maxValue / gridLines) * i;
        const y = padding.top + (chartHeight / gridLines) * i;
        ctx.fillText(Math.round(value).toLocaleString(), padding.left - 10, y);
    }

    // 绘制X轴标签（每10个点显示一个，或最多显示10个）
    ctx.textAlign = 'center';
    ctx.textBaseline = 'top';
    const labelStep = Math.max(1, Math.floor(tokenData.length / 10));
    for (let i = 0; i < tokenData.length; i += labelStep) {
        const x = padding.left + stepX * i;
        if (x <= width - padding.right) {
            ctx.fillText(`#${i + 1}`, x, height - padding.bottom + 10);
        }
    }

    // 绘制折线图（输入Token和输出Token分别用不同颜色）
    // 输入Token - 蓝色
    ctx.strokeStyle = '#1976d2';
    ctx.lineWidth = 2;
    ctx.beginPath();
    
    let hasInputData = false;
    for (let i = 0; i < tokenData.length; i++) {
        const x = padding.left + stepX * i;
        const y = height - padding.bottom - (tokenData[i].inputToken * scaleY);
        
        if (i === 0) {
            ctx.moveTo(x, y);
        } else {
            ctx.lineTo(x, y);
        }
        
        if (tokenData[i].inputToken > 0) {
            hasInputData = true;
        }
    }
    
    if (hasInputData) {
        ctx.stroke();
    }

    // 输出Token - 绿色
    ctx.strokeStyle = '#388e3c';
    ctx.lineWidth = 2;
    ctx.beginPath();
    
    let hasOutputData = false;
    for (let i = 0; i < tokenData.length; i++) {
        const x = padding.left + stepX * i;
        const y = height - padding.bottom - (tokenData[i].outputToken * scaleY);
        
        if (i === 0) {
            ctx.moveTo(x, y);
        } else {
            ctx.lineTo(x, y);
        }
        
        if (tokenData[i].outputToken > 0) {
            hasOutputData = true;
        }
    }
    
    if (hasOutputData) {
        ctx.stroke();
    }

    // 总Token - 紫色
    ctx.strokeStyle = '#9c27b0';
    ctx.lineWidth = 2;
    ctx.beginPath();
    
    let hasTotalData = false;
    for (let i = 0; i < tokenData.length; i++) {
        const x = padding.left + stepX * i;
        const y = height - padding.bottom - (tokenData[i].totalTokens * scaleY);
        
        if (i === 0) {
            ctx.moveTo(x, y);
        } else {
            ctx.lineTo(x, y);
        }
        
        if (tokenData[i].totalTokens > 0) {
            hasTotalData = true;
        }
    }
    
    if (hasTotalData) {
        ctx.stroke();
        
        // 绘制数据点
        ctx.fillStyle = '#9c27b0';
        for (let i = 0; i < tokenData.length; i++) {
            if (tokenData[i].totalTokens > 0) {
                const x = padding.left + stepX * i;
                const y = height - padding.bottom - (tokenData[i].totalTokens * scaleY);
                ctx.beginPath();
                ctx.arc(x, y, 3, 0, Math.PI * 2);
                ctx.fill();
            }
        }
    }

    // 绘制图例
    const legendY = 20;
    const legendX = width - padding.right - 150;
    
    // 输入Token图例
    ctx.strokeStyle = '#1976d2';
    ctx.lineWidth = 2;
    ctx.beginPath();
    ctx.moveTo(legendX, legendY);
    ctx.lineTo(legendX + 20, legendY);
    ctx.stroke();
    ctx.fillStyle = '#333';
    ctx.font = '11px Arial';
    ctx.textAlign = 'left';
    ctx.fillText('输入Token', legendX + 25, legendY + 4);
    
    // 输出Token图例
    ctx.strokeStyle = '#388e3c';
    ctx.beginPath();
    ctx.moveTo(legendX, legendY + 15);
    ctx.lineTo(legendX + 20, legendY + 15);
    ctx.stroke();
    ctx.fillText('输出Token', legendX + 25, legendY + 19);
    
    // 总Token图例
    ctx.strokeStyle = '#9c27b0';
    ctx.beginPath();
    ctx.moveTo(legendX, legendY + 30);
    ctx.lineTo(legendX + 20, legendY + 30);
    ctx.stroke();
    ctx.fillText('总Token', legendX + 25, legendY + 34);

    // 绘制标题
    ctx.fillStyle = '#333';
    ctx.font = 'bold 14px Arial';
    ctx.textAlign = 'center';
    ctx.fillText('压缩事件Token消耗趋势（按事件顺序）', width / 2, 20);
}

function renderCompressionEvents(data) {
    let compressionEvents = [];
    
    // 提取compressionEvents
    if (data.compressionEvents && Array.isArray(data.compressionEvents)) {
        compressionEvents = data.compressionEvents;
    } else if (data.agent && data.agent.memory && data.agent.memory.compressionEvents) {
        compressionEvents = data.agent.memory.compressionEvents;
    }

    const container = document.getElementById('eventsContainer');
    const overviewDiv = document.getElementById('eventsTokenOverview');
    
    if (!compressionEvents || compressionEvents.length === 0) {
        container.innerHTML = 
            '<div class="empty-state"><div class="empty-state-icon">⚠️</div><div>未找到compressionEvents数据</div></div>';
        if (overviewDiv) {
            overviewDiv.style.display = 'none';
        }
        return;
    }

    // 按时间戳排序（从早到晚）
    const sortedEvents = [...compressionEvents].sort((a, b) => (a.timestamp || 0) - (b.timestamp || 0));
    
    // 计算Token消耗统计
    let totalInputTokens = 0;
    let totalOutputTokens = 0;
    let totalTime = 0;
    let tokenData = [];
    let eventCount = 0;

    sortedEvents.forEach((event, index) => {
        if (event.metadata) {
            const inputToken = event.metadata.inputToken || 0;
            const outputToken = event.metadata.outputToken || 0;
            const time = event.metadata.time || 0;
            const totalTokens = inputToken + outputToken;
            
            if (totalTokens > 0) {
                totalInputTokens += inputToken;
                totalOutputTokens += outputToken;
                totalTime += time;
                eventCount++;
            }
            
            tokenData.push({
                index: index,
                inputToken: inputToken,
                outputToken: outputToken,
                totalTokens: totalTokens,
                time: time,
                timestamp: event.timestamp || index
            });
        } else {
            tokenData.push({
                index: index,
                inputToken: 0,
                outputToken: 0,
                totalTokens: 0,
                time: 0,
                timestamp: event.timestamp || index
            });
        }
    });

    // 显示Token消耗总览
    if (overviewDiv && eventCount > 0) {
        overviewDiv.style.display = 'block';
        const totalTokens = totalInputTokens + totalOutputTokens;
        document.getElementById('eventsTotalInputTokens').textContent = totalInputTokens.toLocaleString();
        document.getElementById('eventsTotalOutputTokens').textContent = totalOutputTokens.toLocaleString();
        document.getElementById('eventsTotalTokens').textContent = totalTokens.toLocaleString();
        document.getElementById('eventsAvgTokens').textContent = Math.round(totalTokens / eventCount).toLocaleString();
        document.getElementById('eventsTotalTime').textContent = totalTime.toFixed(2);

        // 保存数据用于重绘
        eventsTokenChartData = { tokenData, totalTokens, maxTokens: Math.max(...tokenData.map(d => d.totalTokens)) };
        
        // 绘制趋势图表
        drawEventsTokenChart(tokenData, totalTokens);
        
        // 监听窗口大小变化，重新绘制图表
        if (!window.eventsTokenChartResizeHandler) {
            window.eventsTokenChartResizeHandler = function() {
                if (eventsTokenChartData) {
                    setTimeout(() => {
                        drawEventsTokenChart(
                            eventsTokenChartData.tokenData,
                            eventsTokenChartData.totalTokens
                        );
                    }, 100);
                }
            };
            window.addEventListener('resize', window.eventsTokenChartResizeHandler);
        }
    } else if (overviewDiv) {
        overviewDiv.style.display = 'none';
        eventsTokenChartData = null;
    }

    container.innerHTML = '';
    
    sortedEvents.forEach((event, index) => {
        // 创建事件卡片
        const eventDiv = document.createElement('div');
        eventDiv.style.marginBottom = '20px';
        eventDiv.style.padding = '16px';
        eventDiv.style.background = '#ffffff';
        eventDiv.style.border = '2px solid #9c27b0';
        eventDiv.style.borderRadius = '8px';
        eventDiv.style.boxShadow = '0 2px 8px rgba(0,0,0,0.1)';
        
        // 事件头部
        const eventHeader = document.createElement('div');
        eventHeader.style.display = 'flex';
        eventHeader.style.justifyContent = 'space-between';
        eventHeader.style.alignItems = 'center';
        eventHeader.style.marginBottom = '12px';
        eventHeader.style.paddingBottom = '8px';
        eventHeader.style.borderBottom = '2px solid #e0e0e0';
        
        // 事件序号和类型
        const eventTypeDiv = document.createElement('div');
        eventTypeDiv.style.display = 'flex';
        eventTypeDiv.style.alignItems = 'center';
        eventTypeDiv.style.gap = '12px';
        
        const eventIndex = document.createElement('span');
        eventIndex.style.fontSize = '16px';
        eventIndex.style.fontWeight = 'bold';
        eventIndex.style.color = '#9c27b0';
        eventIndex.textContent = `#${index + 1}`;
        
        // 定义事件类型的中文名称和颜色
        const eventTypeMap = {
            'TOOL_INVOCATION_COMPRESS': { name: '工具调用压缩', color: '#ff9800' },
            'LARGE_MESSAGE_OFFLOAD_WITH_PROTECTION': { name: '大消息卸载（带保护）', color: '#2196f3' },
            'LARGE_MESSAGE_OFFLOAD': { name: '大消息卸载（无保护）', color: '#03a9f4' },
            'PREVIOUS_ROUND_CONVERSATION_SUMMARY': { name: '上一轮对话摘要', color: '#4caf50' },
            'CURRENT_ROUND_LARGE_MESSAGE_SUMMARY': { name: '当前轮次大消息摘要', color: '#8bc34a' },
            'CURRENT_ROUND_MESSAGE_COMPRESS': { name: '当前轮次消息压缩', color: '#9c27b0' }
        };
        
        const eventTypeInfo = eventTypeMap[event.eventType] || { 
            name: event.eventType || '未知类型', 
            color: '#757575' 
        };
        
        const eventTypeBadge = document.createElement('span');
        eventTypeBadge.style.padding = '6px 12px';
        eventTypeBadge.style.background = eventTypeInfo.color;
        eventTypeBadge.style.color = 'white';
        eventTypeBadge.style.borderRadius = '12px';
        eventTypeBadge.style.fontSize = '12px';
        eventTypeBadge.style.fontWeight = '600';
        eventTypeBadge.textContent = eventTypeInfo.name;
        
        eventTypeDiv.appendChild(eventIndex);
        eventTypeDiv.appendChild(eventTypeBadge);
        
        // 时间戳
        const timestampDiv = document.createElement('div');
        timestampDiv.style.fontSize = '12px';
        timestampDiv.style.color = '#666';
        if (event.timestamp) {
            const date = new Date(event.timestamp);
            timestampDiv.textContent = date.toLocaleString('zh-CN');
        } else {
            timestampDiv.textContent = '时间未知';
        }
        
        eventHeader.appendChild(eventTypeDiv);
        eventHeader.appendChild(timestampDiv);
        eventDiv.appendChild(eventHeader);
        
        // 事件详情
        const eventDetails = document.createElement('div');
        eventDetails.style.display = 'grid';
        eventDetails.style.gridTemplateColumns = 'repeat(auto-fit, minmax(250px, 1fr))';
        eventDetails.style.gap = '16px';
        eventDetails.style.marginTop = '12px';
        
        // Token信息卡片
        const tokenCard = document.createElement('div');
        tokenCard.style.padding = '12px';
        tokenCard.style.background = '#f5f5f5';
        tokenCard.style.borderRadius = '6px';
        tokenCard.style.border = '1px solid #e0e0e0';
        
        const tokenTitle = document.createElement('div');
        tokenTitle.style.fontSize = '13px';
        tokenTitle.style.fontWeight = 'bold';
        tokenTitle.style.color = '#333';
        tokenTitle.style.marginBottom = '8px';
        tokenTitle.textContent = 'Token 使用情况';
        tokenCard.appendChild(tokenTitle);
        
        // 使用metadata中的输入/输出token和耗时信息
        if (event.metadata) {
            const metadataInfo = [];
            if (event.metadata.inputToken !== undefined) {
                metadataInfo.push(`输入Token: <strong style="color: #1976d2;">${event.metadata.inputToken.toLocaleString()}</strong>`);
            }
            if (event.metadata.outputToken !== undefined) {
                metadataInfo.push(`输出Token: <strong style="color: #388e3c;">${event.metadata.outputToken.toLocaleString()}</strong>`);
            }
            if (event.metadata.time !== undefined) {
                metadataInfo.push(`耗时: <strong style="color: #f57c00;">${event.metadata.time.toFixed(2)}</strong> 秒`);
            }
            
            if (metadataInfo.length > 0) {
                const metadataDiv = document.createElement('div');
                metadataDiv.style.fontSize = '12px';
                metadataDiv.style.color = '#666';
                metadataDiv.innerHTML = metadataInfo.join(' | ');
                tokenCard.appendChild(metadataDiv);
            } else {
                // 如果没有metadata信息，显示提示
                const noDataDiv = document.createElement('div');
                noDataDiv.style.fontSize = '12px';
                noDataDiv.style.color = '#999';
                noDataDiv.textContent = '无Token使用数据';
                tokenCard.appendChild(noDataDiv);
            }
        } else {
            // 如果没有metadata，显示提示
            const noDataDiv = document.createElement('div');
            noDataDiv.style.fontSize = '12px';
            noDataDiv.style.color = '#999';
            noDataDiv.textContent = '无Token使用数据';
            tokenCard.appendChild(noDataDiv);
        }
        
        eventDetails.appendChild(tokenCard);
        
        // 消息统计卡片
        const messageCard = document.createElement('div');
        messageCard.style.padding = '12px';
        messageCard.style.background = '#f5f5f5';
        messageCard.style.borderRadius = '6px';
        messageCard.style.border = '1px solid #e0e0e0';
        
        const messageTitle = document.createElement('div');
        messageTitle.style.fontSize = '13px';
        messageTitle.style.fontWeight = 'bold';
        messageTitle.style.color = '#333';
        messageTitle.style.marginBottom = '8px';
        messageTitle.textContent = '消息统计';
        messageCard.appendChild(messageTitle);
        
        const messageCount = document.createElement('div');
        messageCount.style.fontSize = '12px';
        messageCount.style.color = '#666';
        messageCount.innerHTML = `压缩消息数: <strong>${event.compressedMessageCount || 0}</strong> 条`;
        messageCard.appendChild(messageCount);
        
        eventDetails.appendChild(messageCard);
        
        // 消息ID信息卡片
        const idCard = document.createElement('div');
        idCard.style.padding = '12px';
        idCard.style.background = '#f5f5f5';
        idCard.style.borderRadius = '6px';
        idCard.style.border = '1px solid #e0e0e0';
        
        const idTitle = document.createElement('div');
        idTitle.style.fontSize = '13px';
        idTitle.style.fontWeight = 'bold';
        idTitle.style.color = '#333';
        idTitle.style.marginBottom = '8px';
        idTitle.textContent = '消息ID';
        idCard.appendChild(idTitle);
        
        if (event.previousMessageId) {
            const prevId = document.createElement('div');
            prevId.style.fontSize = '11px';
            prevId.style.color = '#666';
            prevId.style.marginBottom = '4px';
            prevId.style.wordBreak = 'break-all';
            prevId.innerHTML = `前一个: <code style="background: #e0e0e0; padding: 2px 4px; border-radius: 3px;">${event.previousMessageId}</code>`;
            idCard.appendChild(prevId);
        }
        
        if (event.compressedMessageId) {
            const compressedId = document.createElement('div');
            compressedId.style.fontSize = '11px';
            compressedId.style.color = '#666';
            compressedId.style.marginBottom = '4px';
            compressedId.style.wordBreak = 'break-all';
            compressedId.innerHTML = `压缩后: <code style="background: #fff3cd; padding: 2px 4px; border-radius: 3px;">${event.compressedMessageId}</code>`;
            idCard.appendChild(compressedId);
        }
        
        if (event.nextMessageId) {
            const nextId = document.createElement('div');
            nextId.style.fontSize = '11px';
            nextId.style.color = '#666';
            nextId.style.wordBreak = 'break-all';
            nextId.innerHTML = `下一个: <code style="background: #e0e0e0; padding: 2px 4px; border-radius: 3px;">${event.nextMessageId}</code>`;
            idCard.appendChild(nextId);
        }
        
        eventDetails.appendChild(idCard);
        
        eventDiv.appendChild(eventDetails);
        container.appendChild(eventDiv);
    });
}
