// message-utils.js
// 依赖: common.js (extractUuidFromText, getOriginalContentFromOffload, calculateTextLength, getMessagePreview, createContentItem, getOffloadContext, calculateTokenUsageRanking)

function createMessageElement(msg, index, data, tokenRanking) {
    const div = document.createElement('div');
    
    // 确定消息类型和样式
    let typeClass = 'message-type-system';
    let typeBadgeClass = 'type-badge-system';
    let typeLabel = msg.role || 'UNKNOWN';
    let hasCompressed = false;
    
    if (msg.name === 'long_term_memory') {
        typeClass = 'message-type-long-term-memory';
        typeBadgeClass = 'type-badge-long-term-memory';
        typeLabel = 'LONG_TERM_MEMORY';
    } else if (msg.role === 'USER') {
        typeClass = 'message-type-user';
        typeBadgeClass = 'type-badge-user';
        typeLabel = 'USER';
    } else if (msg.role === 'ASSISTANT') {
        // 通过metadata中的_compress_meta判断是否被压缩
        hasCompressed = !!(msg.metadata && msg.metadata._compress_meta);
        
        // 如果name是assistant且有_compress_meta，显示为compressed类型
        if (msg.name === 'assistant' && hasCompressed) {
            typeClass = 'message-type-compressed';
            typeBadgeClass = 'type-badge-compressed';
            typeLabel = 'Compressed';
        } else {
            // 检查是否包含tool_use
            let hasToolUse = false;
            if (msg.content && Array.isArray(msg.content)) {
                hasToolUse = msg.content.some(item => item.type === 'tool_use');
            }
            
            typeClass = 'message-type-assistant';
            typeBadgeClass = 'type-badge-assistant';
            
            if (hasToolUse) {
                typeLabel = 'Tool use';
                typeBadgeClass = 'type-badge-tool-use';
            } else {
                typeLabel = 'assistant-reply';
            }
        }
    } else if (msg.role === 'TOOL') {
        typeClass = 'message-type-tool';
        typeBadgeClass = 'type-badge-tool';
        typeLabel = 'Tool Result';
    } else if (msg.role === 'SYSTEM') {
        typeClass = 'message-type-system';
        typeBadgeClass = 'type-badge-system';
        typeLabel = 'SYSTEM';
    }

    div.className = `message ${typeClass} collapsed`;
    if (hasCompressed) {
        div.classList.add('message-compressed');
    }
    
    // 为消息元素添加唯一ID，用于定位
    let messageId = msg.id;
    if (!messageId) {
        const role = msg.role || 'unknown';
        const name = msg.name || 'unknown';
        const timestamp = msg.timestamp || '';
        const contentHash = msg.content ? JSON.stringify(msg.content).substring(0, 100) : '';
        messageId = `msg_${role}_${name}_${timestamp}_${contentHash}`.replace(/[^a-zA-Z0-9_]/g, '_');
    }
    div.id = `message-${messageId}`;

    // 提取thinking内容和工具名称
    let thinkingContent = null;
    let filteredContent = [];
    let compressedUuid = null;
    let originalContent = null;
    let toolName = null;
    
    // 如果消息被压缩，从metadata中提取压缩信息
    if (hasCompressed && msg.metadata && msg.metadata._compress_meta) {
        const compressMeta = msg.metadata._compress_meta;
        compressedUuid = compressMeta.offloaduuid || null;
        
        // 从offloadContext获取原始内容
        if (compressedUuid && data) {
            const offloadContext = getOffloadContext(data);
            if (offloadContext) {
                originalContent = getOriginalContentFromOffload(offloadContext, compressedUuid);
            }
        }
    }
    
    if (msg.content && Array.isArray(msg.content)) {
        msg.content.forEach(item => {
            if (item.type === 'thinking') {
                thinkingContent = item.thinking || '';
            } else if (item.type === 'tool_use' || item.type === 'tool_result') {
                // 提取工具名称
                if (!toolName && item.name) {
                    toolName = item.name;
                }
                filteredContent.push(item);
            } else {
                filteredContent.push(item);
            }
        });
    }

    // 消息头部
    const header = document.createElement('div');
    header.className = 'message-header';
    
    const headerLeft = document.createElement('div');
    headerLeft.className = 'message-header-left';
    
    const typeBadge = document.createElement('span');
    typeBadge.className = `message-type ${typeBadgeClass}`;
    typeBadge.textContent = typeLabel;
    
    headerLeft.appendChild(typeBadge);
    
    // 如果是工具消息，在headerLeft中添加工具名称
    if (toolName && (msg.role === 'TOOL' || typeLabel === 'Tool use')) {
        const toolNameSpan = document.createElement('span');
        toolNameSpan.style.marginLeft = '12px';
        toolNameSpan.style.fontSize = '13px';
        toolNameSpan.style.fontWeight = '600';
        toolNameSpan.style.color = '#333';
        toolNameSpan.textContent = toolName;
        headerLeft.appendChild(toolNameSpan);
    }
    
    const headerRight = document.createElement('div');
    headerRight.className = 'message-header-right';
    
    // 如果消息包含compressed内容，显示compressed标识
    if (hasCompressed) {
        // 如果带有uuid，显示压缩信息（一行显示）
        if (compressedUuid) {
            // 计算压缩前后的字符数
            let compressedLength = 0;
            if (msg.content && Array.isArray(msg.content)) {
                // 直接计算msg.content的长度（内容项数组）
                msg.content.forEach(item => {
                    if (item.type === 'text' && item.text) {
                        compressedLength += item.text.length;
                    } else if (item.type === 'thinking' && item.thinking) {
                        compressedLength += item.thinking.length;
                    } else if (item.type === 'tool_use' && item.input) {
                        compressedLength += JSON.stringify(item.input).length;
                    } else if (item.type === 'tool_result' && item.output) {
                        if (Array.isArray(item.output)) {
                            item.output.forEach(o => {
                                if (o.type === 'text' && o.text) {
                                    compressedLength += o.text.length;
                                } else {
                                    compressedLength += JSON.stringify(o).length;
                                }
                            });
                        } else {
                            compressedLength += JSON.stringify(item.output).length;
                        }
                    }
                });
            }
            
            let originalLength = 0;
            if (originalContent && Array.isArray(originalContent)) {
                // originalContent是消息数组，使用calculateTextLength
                originalLength = calculateTextLength(originalContent);
            }
            
            const compressionRatio = originalLength > 0 ? ((1 - compressedLength / originalLength) * 100).toFixed(1) : 0;
            
            // 创建压缩信息（一行显示）
            const compressedInfo = document.createElement('span');
            compressedInfo.className = 'compressed-info';
            compressedInfo.style.marginLeft = '8px';
            compressedInfo.style.fontSize = '11px';
            compressedInfo.style.color = '#666';
            compressedInfo.style.whiteSpace = 'nowrap';
            
            // 添加查看原文按钮（如果有原始内容）
            let viewOriginalBtn = null;
            if (originalContent && originalContent.length > 0) {
                viewOriginalBtn = document.createElement('button');
                viewOriginalBtn.className = 'view-original-btn';
                viewOriginalBtn.textContent = '查看原文';
                viewOriginalBtn.style.marginLeft = '8px';
                viewOriginalBtn.style.padding = '2px 8px';
                viewOriginalBtn.style.fontSize = '10px';
                viewOriginalBtn.style.background = '#007bff';
                viewOriginalBtn.style.color = 'white';
                viewOriginalBtn.style.border = 'none';
                viewOriginalBtn.style.borderRadius = '3px';
                viewOriginalBtn.style.cursor = 'pointer';
                viewOriginalBtn.style.transition = 'background 0.2s';
                viewOriginalBtn.style.verticalAlign = 'middle';
                
                viewOriginalBtn.onmouseenter = function() {
                    this.style.background = '#0056b3';
                };
                viewOriginalBtn.onmouseleave = function() {
                    this.style.background = '#007bff';
                };
                
                // 创建原文显示区域（初始隐藏）
                const originalContentDiv = document.createElement('div');
                originalContentDiv.className = 'original-content-modal';
                originalContentDiv.style.display = 'none';
                originalContentDiv.style.position = 'fixed';
                originalContentDiv.style.top = '50%';
                originalContentDiv.style.left = '50%';
                originalContentDiv.style.transform = 'translate(-50%, -50%)';
                originalContentDiv.style.width = '80%';
                originalContentDiv.style.maxWidth = '900px';
                originalContentDiv.style.maxHeight = '80vh';
                originalContentDiv.style.background = 'white';
                originalContentDiv.style.border = '2px solid #007bff';
                originalContentDiv.style.borderRadius = '8px';
                originalContentDiv.style.boxShadow = '0 4px 20px rgba(0,0,0,0.3)';
                originalContentDiv.style.zIndex = '10001';
                originalContentDiv.style.overflow = 'hidden';
                originalContentDiv.style.flexDirection = 'column';
                
                // 标题栏
                const modalHeader = document.createElement('div');
                modalHeader.style.padding = '12px 16px';
                modalHeader.style.background = '#007bff';
                modalHeader.style.color = 'white';
                modalHeader.style.display = 'flex';
                modalHeader.style.justifyContent = 'space-between';
                modalHeader.style.alignItems = 'center';
                modalHeader.innerHTML = `
                    <div style="font-weight: bold;">原始内容 (UUID: ${compressedUuid})</div>
                    <button class="close-original-btn" style="background: transparent; border: none; color: white; font-size: 20px; cursor: pointer; padding: 0; width: 24px; height: 24px; line-height: 24px;">&times;</button>
                `;
                
                // 内容区域
                const modalContent = document.createElement('div');
                modalContent.style.padding = '16px';
                modalContent.style.overflowY = 'auto';
                modalContent.style.flex = '1';
                modalContent.style.fontSize = '12px';
                modalContent.style.lineHeight = '1.6';
                modalContent.style.whiteSpace = 'pre-wrap';
                modalContent.style.wordBreak = 'break-word';
                modalContent.style.color = '#333';
                
                // 格式化原始内容
                let originalText = '';
                originalContent.forEach((origMsg, idx) => {
                    if (idx > 0) originalText += '\n' + '='.repeat(80) + '\n\n';
                    
                    // 消息头部信息
                    originalText += `[消息 ${idx + 1}]\n`;
                    originalText += `Role: ${origMsg.role || 'N/A'}\n`;
                    originalText += `Name: ${origMsg.name || 'N/A'}\n`;
                    originalText += `ID: ${origMsg.id || 'N/A'}\n`;
                    originalText += `Timestamp: ${origMsg.timestamp || 'N/A'}\n`;
                    originalText += '\n';
                    
                    if (origMsg.content && Array.isArray(origMsg.content)) {
                        origMsg.content.forEach((item, itemIdx) => {
                            if (itemIdx > 0) originalText += '\n---\n\n';
                            
                            if (item.type === 'text' && item.text) {
                                originalText += `[Text]\n${item.text}\n`;
                            } else if (item.type === 'thinking' && item.thinking) {
                                originalText += `[Thinking]\n${item.thinking}\n`;
                            } else if (item.type === 'tool_use') {
                                originalText += `[Tool Use]\n`;
                                originalText += `Name: ${item.name || 'unknown'}\n`;
                                originalText += `ID: ${item.id || 'N/A'}\n`;
                                if (item.input) {
                                    originalText += `Input:\n${JSON.stringify(item.input, null, 2)}\n`;
                                }
                            } else if (item.type === 'tool_result') {
                                originalText += `[Tool Result]\n`;
                                originalText += `Name: ${item.name || 'unknown'}\n`;
                                originalText += `ID: ${item.id || 'N/A'}\n`;
                                if (item.output) {
                                    originalText += `Output:\n`;
                                    if (Array.isArray(item.output)) {
                                        item.output.forEach(o => {
                                            if (o.type === 'text') {
                                                originalText += o.text + '\n';
                                            } else {
                                                originalText += JSON.stringify(o, null, 2) + '\n';
                                            }
                                        });
                                    } else {
                                        originalText += JSON.stringify(item.output, null, 2) + '\n';
                                    }
                                }
                            } else {
                                originalText += `[${item.type || 'Unknown'}]\n${JSON.stringify(item, null, 2)}\n`;
                            }
                        });
                    }
                    originalText += '\n';
                });
                
                modalContent.textContent = originalText || '无原始内容';
                
                // 创建背景遮罩
                const modalOverlay = document.createElement('div');
                modalOverlay.className = 'modal-overlay';
                modalOverlay.style.position = 'fixed';
                modalOverlay.style.top = '0';
                modalOverlay.style.left = '0';
                modalOverlay.style.width = '100%';
                modalOverlay.style.height = '100%';
                modalOverlay.style.background = 'rgba(0, 0, 0, 0.5)';
                modalOverlay.style.zIndex = '10000';
                modalOverlay.style.display = 'none';
                
                // 关闭按钮事件
                const closeBtn = modalHeader.querySelector('.close-original-btn');
                const closeModal = function() {
                    originalContentDiv.style.display = 'none';
                    modalOverlay.style.display = 'none';
                    if (originalContentDiv.parentNode) {
                        document.body.removeChild(originalContentDiv);
                    }
                    if (modalOverlay.parentNode) {
                        document.body.removeChild(modalOverlay);
                    }
                };
                
                closeBtn.onclick = closeModal;
                modalOverlay.onclick = closeModal;
                
                // 点击查看原文按钮
                viewOriginalBtn.onclick = function(e) {
                    e.stopPropagation();
                    if (!originalContentDiv.parentNode) {
                        originalContentDiv.appendChild(modalHeader);
                        originalContentDiv.appendChild(modalContent);
                        document.body.appendChild(modalOverlay);
                        document.body.appendChild(originalContentDiv);
                    }
                    modalOverlay.style.display = 'block';
                    originalContentDiv.style.display = 'flex';
                };
                
                msg._originalContent = originalContent;
            }
            
            compressedInfo.innerHTML = `
                <strong>UUID:</strong> ${compressedUuid} | 
                压缩后: ${compressedLength.toLocaleString()} 字符 | 
                原始内容: ${originalLength.toLocaleString()} 字符 | 
                <span style="font-weight: bold; color: #856404;">压缩率: ${compressionRatio}%</span>
            `;
            
            headerRight.appendChild(compressedInfo);
            if (viewOriginalBtn) {
                headerRight.appendChild(viewOriginalBtn);
            }
        }
    }
    
    // 如果消息的metadata中有_chat_usage，显示usage信息图标
    const chatUsage = (msg.metadata && msg.metadata._chat_usage) || msg._chat_usage;
    if (chatUsage) {
        const usageIcon = document.createElement('div');
        usageIcon.className = 'usage-info-icon';
        usageIcon.textContent = 'i';
        usageIcon.title = 'Token使用信息';
        
        // 根据token使用量排名设置颜色
        // 生成与统计时相同的消息ID
        let messageId = msg.id;
        if (!messageId) {
            const role = msg.role || 'unknown';
            const name = msg.name || 'unknown';
            const timestamp = msg.timestamp || '';
            // 使用消息内容的前100个字符作为hash（与统计时保持一致）
            const contentHash = msg.content ? JSON.stringify(msg.content).substring(0, 100) : '';
            messageId = `${role}_${name}_${timestamp}_${contentHash}`;
        }
        
        const rank = tokenRanking ? tokenRanking.get(messageId) : null;
        
        if (rank === 1) {
            // Top 1: 红色
            usageIcon.style.background = '#dc3545';
            usageIcon.style.color = '#fff';
        } else if (rank === 10) {
            // Top 10: 灰色
            usageIcon.style.background = '#6c757d';
            usageIcon.style.color = '#fff';
        } else if (rank && rank >= 2 && rank <= 9) {
            // Top 2-9: 从红色渐变到灰色
            const ratio = (rank - 1) / 9; // 0到1之间的比例
            const red = Math.round(220 - (220 - 108) * ratio); // 220 -> 108
            const green = Math.round(53 - (53 - 117) * ratio); // 53 -> 117
            const blue = Math.round(69 - (69 - 125) * ratio); // 69 -> 125
            usageIcon.style.background = `rgb(${red}, ${green}, ${blue})`;
            usageIcon.style.color = '#fff';
        } else {
            // Top 10以外: 淡绿色
            usageIcon.style.background = '#90ee90';
            usageIcon.style.color = '#333';
        }
        
        const usageTooltip = document.createElement('div');
        usageTooltip.className = 'usage-tooltip';
        
        // 格式化_chat_usage数据
        const usage = chatUsage;
        // 确保背景色和文字颜色正确设置
        usageTooltip.style.background = '#1a1a1a';
        usageTooltip.style.color = '#fff';
        usageTooltip.innerHTML = `
            <div style="font-weight: bold; margin-bottom: 8px; color: #fff !important;">Token使用信息</div>
            <div style="color: #fff !important;"><strong style="color: #fff !important;">inputTokens:</strong> ${usage.inputTokens || 0}</div>
            <div style="color: #fff !important;"><strong style="color: #fff !important;">outputTokens:</strong> ${usage.outputTokens || 0}</div>
            <div style="color: #fff !important;"><strong style="color: #fff !important;">totalTokens:</strong> ${usage.totalTokens || 0}</div>
            <div style="color: #fff !important;"><strong style="color: #fff !important;">time:</strong> ${usage.time || 0} 秒</div>
        `;
        
        document.body.appendChild(usageTooltip);
        
        // 动态定位tooltip
        usageIcon.addEventListener('mouseenter', function(e) {
            const iconRect = usageIcon.getBoundingClientRect();
            
            // 先显示tooltip以获取尺寸（但不可见）
            usageTooltip.style.display = 'block';
            usageTooltip.style.visibility = 'hidden';
            const tooltipRect = usageTooltip.getBoundingClientRect();
            usageTooltip.style.visibility = 'visible';
            
            // 计算位置：在图标上方，右对齐
            let left = iconRect.right - tooltipRect.width;
            let top = iconRect.top - tooltipRect.height - 8;
            
            // 如果左侧超出屏幕，则左对齐到图标
            if (left < 10) {
                left = iconRect.left;
            }
            
            // 如果上方空间不够，显示在下方
            if (top < 10) {
                top = iconRect.bottom + 8;
            }
            
            usageTooltip.style.left = left + 'px';
            usageTooltip.style.top = top + 'px';
        });
        
        usageIcon.addEventListener('mouseleave', function() {
            usageTooltip.style.display = 'none';
        });
        
        headerRight.appendChild(usageIcon);
    }
    
    // 如果是ASSISTANT消息且有thinking内容，显示thinking标识
    if (msg.role === 'ASSISTANT' && thinkingContent) {
        const thinkingBadge = document.createElement('div');
        thinkingBadge.className = 'thinking-badge';
        thinkingBadge.textContent = 'Thinking';
        
        const tooltip = document.createElement('div');
        tooltip.className = 'thinking-tooltip';
        tooltip.textContent = thinkingContent;
        document.body.appendChild(tooltip);
        
        // 显示tooltip的函数
        function showTooltip() {
            const badgeRect = thinkingBadge.getBoundingClientRect();
            
            // 先显示tooltip以获取尺寸（但不可见）
            tooltip.style.display = 'block';
            tooltip.style.visibility = 'hidden';
            tooltip.style.width = '500px';
            tooltip.style.maxWidth = '90vw';
            const tooltipRect = tooltip.getBoundingClientRect();
            tooltip.style.visibility = 'visible';
            
            // 计算位置：优先显示在badge的下方，紧挨着（0px间隙，甚至稍微重叠）
            let left = badgeRect.left;
            let top = badgeRect.bottom;
            
            // 如果右侧超出屏幕，调整left位置
            if (left + tooltipRect.width > window.innerWidth) {
                left = window.innerWidth - tooltipRect.width - 12;
                if (left < 12) {
                    left = 12;
                }
            }
            
            // 如果下方空间不够，显示在上方（下边框紧挨着badge顶部，0px间隙）
            if (top + tooltipRect.height > window.innerHeight) {
                // 显示在上方时，浮层的下边框紧挨着badge的顶部
                top = badgeRect.top - tooltipRect.height;
                if (top < 12) {
                    // 如果上方空间也不够，至少保证顶部不超出屏幕
                    top = 12;
                }
            }
            
            tooltip.style.left = left + 'px';
            tooltip.style.top = top + 'px';
            tooltip.style.width = Math.min(500, window.innerWidth - left - 24) + 'px';
        }
        
        let hideTimeout = null;
        
        // badge的鼠标事件
        thinkingBadge.addEventListener('mouseenter', function(e) {
            // 清除隐藏定时器
            if (hideTimeout) {
                clearTimeout(hideTimeout);
                hideTimeout = null;
            }
            showTooltip();
        });
        
        thinkingBadge.addEventListener('mouseleave', function(e) {
            // 延迟隐藏，给鼠标移动到tooltip的时间（增加到500ms）
            hideTimeout = setTimeout(() => {
                tooltip.style.display = 'none';
            }, 500);
        });
        
        // tooltip的鼠标事件（允许鼠标移到tooltip上）
        tooltip.addEventListener('mouseenter', function() {
            // 清除隐藏定时器
            if (hideTimeout) {
                clearTimeout(hideTimeout);
                hideTimeout = null;
            }
            // 鼠标进入tooltip时保持显示
            tooltip.style.display = 'block';
        });
        
        tooltip.addEventListener('mouseleave', function() {
            // 鼠标离开tooltip时，延迟隐藏（给时间移回badge）
            hideTimeout = setTimeout(() => {
                tooltip.style.display = 'none';
            }, 200);
        });
        
        headerRight.appendChild(thinkingBadge);
    }
    
    const timestamp = document.createElement('span');
    timestamp.className = 'message-timestamp';
    timestamp.textContent = msg.timestamp || '';
    headerRight.appendChild(timestamp);

    header.appendChild(headerLeft);
    header.appendChild(headerRight);
    div.appendChild(header);

    // 消息预览（单行显示）
    const previewDiv = document.createElement('div');
    previewDiv.className = 'message-preview';
    
    // 检查是否是工具消息，需要特殊处理
    let isToolMessage = false;
    let toolPreviewContent = null;
    if (msg.content && Array.isArray(msg.content)) {
        const toolItem = msg.content.find(item => item.type === 'tool_use' || item.type === 'tool_result');
        if (toolItem) {
            isToolMessage = true;
            const toolName = toolItem.name || 'unknown';
            
            if (toolItem.type === 'tool_use') {
                
                let secondLine = '参数：  ';
                if (toolItem.input && typeof toolItem.input === 'object') {
                    const params = [];
                    for (const [key, value] of Object.entries(toolItem.input)) {
                        const valueStr = typeof value === 'string' ? value : JSON.stringify(value);
                        params.push(`${key}:${valueStr}`);
                    }
                    secondLine += params.join('  ');
                } else if (toolItem.input) {
                    secondLine += JSON.stringify(toolItem.input);
                } else {
                    secondLine += '无参数';
                }
                toolPreviewContent = `${secondLine}`;
            } else if (toolItem.type === 'tool_result') {
                
                let secondLine = '结果：  ';
                if (toolItem.output) {
                    if (Array.isArray(toolItem.output)) {
                        const returns = toolItem.output.map(o => {
                            if (o.type === 'text' && o.text) {
                                return o.text;
                            } else {
                                return JSON.stringify(o);
                            }
                        }).join(' ');
                        secondLine += returns;
                    } else {
                        const returnStr = typeof toolItem.output === 'string' 
                            ? toolItem.output 
                            : JSON.stringify(toolItem.output);
                        secondLine += returnStr;
                    }
                } else {
                    secondLine += '无返回内容';
                }
                toolPreviewContent = `${secondLine}`;
            }
        }
    }
    
    if (isToolMessage && toolPreviewContent) {
        // 工具消息使用特殊样式，显示两行
        previewDiv.style.whiteSpace = 'pre-wrap';
        previewDiv.style.fontFamily = 'Monaco, "Courier New", monospace';
        previewDiv.style.fontSize = '12px';
        previewDiv.style.lineHeight = '1.6';
        previewDiv.style.overflow = 'visible';
        previewDiv.style.textOverflow = 'clip';
        previewDiv.style.maxHeight = 'none';
        previewDiv.style.padding = '4px 0';
        
        // 创建两行内容
        const lines = toolPreviewContent.split('\n');
        const firstLine = document.createElement('div');
        firstLine.style.fontWeight = 'bold';
        firstLine.style.marginBottom = '4px';
        firstLine.textContent = lines[0];
        
        const secondLine = document.createElement('div');
        secondLine.style.color = '#666';
        
        // 获取完整内容
        const fullContent = lines[1] || '';
        const maxLength = 100; // 最大显示长度
        
        // 如果内容超过最大长度，截断并添加省略号
        if (fullContent.length > maxLength) {
            const truncatedContent = fullContent.substring(0, maxLength) + '...';
            secondLine.textContent = truncatedContent;
            secondLine.style.cursor = 'help';
            secondLine.title = ''; // 不使用默认title，使用自定义tooltip
            
            // 创建tooltip显示完整内容
            const tooltip = document.createElement('div');
            tooltip.className = 'tool-param-tooltip';
            tooltip.style.position = 'fixed';
            tooltip.style.background = '#1a1a1a';
            tooltip.style.color = '#fff';
            tooltip.style.border = '2px solid #444';
            tooltip.style.padding = '8px 12px';
            tooltip.style.borderRadius = '8px';
            tooltip.style.fontSize = '12px';
            tooltip.style.fontFamily = 'Monaco, "Courier New", monospace';
            tooltip.style.maxWidth = '600px';
            tooltip.style.maxHeight = '400px';
            tooltip.style.overflow = 'auto';
            tooltip.style.whiteSpace = 'pre-wrap';
            tooltip.style.wordBreak = 'break-word';
            tooltip.style.zIndex = '10002';
            tooltip.style.boxShadow = '0 4px 12px rgba(0,0,0,0.5)';
            tooltip.style.display = 'none';
            tooltip.textContent = fullContent;
            document.body.appendChild(tooltip);
            
            // 添加hover事件
            secondLine.addEventListener('mouseenter', function(e) {
                const rect = secondLine.getBoundingClientRect();
                
                // 先显示tooltip以获取尺寸（但不可见）
                tooltip.style.display = 'block';
                tooltip.style.visibility = 'hidden';
                const tooltipRect = tooltip.getBoundingClientRect();
                tooltip.style.visibility = 'visible';
                
                // 计算位置：显示在元素下方或上方
                let left = rect.left;
                let top = rect.bottom + 8;
                
                // 如果右侧超出屏幕，调整left
                if (left + tooltipRect.width > window.innerWidth) {
                    left = window.innerWidth - tooltipRect.width - 12;
                    if (left < 12) {
                        left = 12;
                    }
                }
                
                // 如果下方空间不够，显示在上方
                if (top + tooltipRect.height > window.innerHeight) {
                    top = rect.top - tooltipRect.height - 8;
                    if (top < 12) {
                        top = 12;
                    }
                }
                
                tooltip.style.left = left + 'px';
                tooltip.style.top = top + 'px';
            });
            
            secondLine.addEventListener('mouseleave', function() {
                tooltip.style.display = 'none';
            });
        } else {
            // 内容不长，直接显示
            secondLine.textContent = fullContent;
        }
        
        previewDiv.innerHTML = '';
        previewDiv.appendChild(firstLine);
        previewDiv.appendChild(secondLine);
    } else {
        previewDiv.textContent = getMessagePreview(msg);
    }
    
    div.appendChild(previewDiv);

    // 创建消息详情浮层（独立于消息，不改变列表布局）
    const detailDiv = document.createElement('div');
    detailDiv.className = 'message-detail';
    detailDiv.id = `detail-${msg.id || index}`;
    const contentDiv = document.createElement('div');
    contentDiv.className = 'message-content';
    
    // 简化显示：只显示预览文本的完整版本，不显示结构化内容
    const fullPreviewText = getFullMessagePreview(msg);
    contentDiv.textContent = fullPreviewText;
    contentDiv.style.whiteSpace = 'pre-wrap';
    contentDiv.style.wordBreak = 'break-word';
    contentDiv.style.fontSize = '13px';
    contentDiv.style.lineHeight = '1.6';
    contentDiv.style.color = '#e0e0e0';

    detailDiv.appendChild(contentDiv);
            
    // 将详情浮层添加到body，而不是消息内部
    document.body.appendChild(detailDiv);
    
    let hideTimeout = null;
    
    // 只在预览内容区域hover时显示详情浮层
    previewDiv.addEventListener('mouseenter', function(e) {
        if (hideTimeout) {
            clearTimeout(hideTimeout);
            hideTimeout = null;
        }
        
        const rect = previewDiv.getBoundingClientRect();
        const messageRect = div.getBoundingClientRect();
        
        // 先显示浮层以获取其尺寸（但不可见）
        detailDiv.style.display = 'block';
        detailDiv.style.visibility = 'hidden';
        detailDiv.style.width = Math.min(800, window.innerWidth - 24) + 'px';
        const detailRect = detailDiv.getBoundingClientRect();
        detailDiv.style.visibility = 'visible';
        
        // 计算浮层位置：优先显示在当前消息的下方
        let left = messageRect.left;
        let top = messageRect.bottom + 8;
        let showAbove = false;
        
        // 如果右侧超出屏幕，调整left位置
        if (left + detailRect.width > window.innerWidth) {
            left = window.innerWidth - detailRect.width - 12;
            if (left < 12) {
                left = 12;
            }
        }
        
        // 如果下方空间不够，显示在上方（下边框挨着消息顶部）
        if (top + detailRect.height > window.innerHeight) {
            showAbove = true;
            // 显示在上方时，浮层的下边框挨着消息的顶部
            top = messageRect.top - detailRect.height - 8;
            if (top < 12) {
                // 如果上方空间也不够，至少保证顶部不超出屏幕
                top = 12;
            }
        }
        
        detailDiv.style.left = left + 'px';
        detailDiv.style.top = top + 'px';
        detailDiv.style.width = Math.min(800, window.innerWidth - left - 24) + 'px';
        detailDiv.classList.add('message-detail-hover');
    });
    
    previewDiv.addEventListener('mouseleave', function(e) {
        // 延迟隐藏，允许鼠标移动到浮层上
        hideTimeout = setTimeout(function() {
            detailDiv.style.display = 'none';
            detailDiv.classList.remove('message-detail-hover');
        }, 200);
    });
    
    // 允许鼠标在浮层上时保持显示
    detailDiv.addEventListener('mouseenter', function() {
        if (hideTimeout) {
            clearTimeout(hideTimeout);
            hideTimeout = null;
        }
        detailDiv.style.display = 'block';
        detailDiv.classList.add('message-detail-hover');
    });
    
    detailDiv.addEventListener('mouseleave', function() {
        detailDiv.style.display = 'none';
        detailDiv.classList.remove('message-detail-hover');
    });

    return div;
}
