// original-memory.js
// 依赖: common.js, message-utils.js

// 存储Token数据，用于窗口大小调整时重绘
let originalTokenChartData = null;
// 存储推理耗时数据，用于窗口大小调整时重绘
let originalTimeChartData = null;

function renderOriginalMemory(data, tokenRanking) {
    let originalMemory = [];
    if (data.agent && data.agent.memory && data.agent.memory.originalMemoryStorage) {
        originalMemory = data.agent.memory.originalMemoryStorage;
    } else if (data.originalMemoryStorage) {
        originalMemory = data.originalMemoryStorage;
    }

    const container = document.getElementById('originalMessagesContainer');
    const overviewDiv = document.getElementById('originalTokenOverview');
    
    if (!originalMemory || originalMemory.length === 0) {
        container.innerHTML = 
            '<div class="empty-state"><div class="empty-state-icon">⚠️</div><div>未找到originalMemoryStorage数据</div></div>';
        if (overviewDiv) {
            overviewDiv.style.display = 'none';
        }
        return;
    }

    // 计算Token消耗统计
    let totalTokens = 0;
    let tokenData = [];
    let maxTokens = 0;
    let tokenCount = 0;

    // 计算推理耗时统计
    let timeData = [];
    let maxTime = 0;
    let totalInferenceTime = 0; // 推理总耗时
    let timeCount = 0;

    // 计算会话总时长（需要减去Agent回复和用户再次回复的时间差）
    let sessionDuration = 0;
    let firstTimestamp = null;
    let lastTimestamp = null;
    let agentReplyToUserGap = 0; // Agent回复到下一个USER消息的时间差总和

    originalMemory.forEach((msg, index) => {
        const chatUsage = (msg.metadata && msg.metadata._chat_usage) || msg._chat_usage;
        if (chatUsage && chatUsage.totalTokens) {
            const tokens = chatUsage.totalTokens;
            totalTokens += tokens;
            tokenData.push({
                index: index,
                tokens: tokens,
                timestamp: msg.timestamp || index
            });
            maxTokens = Math.max(maxTokens, tokens);
            tokenCount++;
        } else {
            tokenData.push({
                index: index,
                tokens: 0,
                timestamp: msg.timestamp || index
            });
        }

        // 收集推理耗时数据
        if (chatUsage && chatUsage.time) {
            const time = chatUsage.time;
            timeData.push({
                index: index,
                time: time,
                timestamp: msg.timestamp || index
            });
            maxTime = Math.max(maxTime, time);
            totalInferenceTime += time;
            timeCount++;
        } else {
            timeData.push({
                index: index,
                time: 0,
                timestamp: msg.timestamp || index
            });
        }

        // 计算时间戳
        if (msg.timestamp) {
            const timestamp = typeof msg.timestamp === 'string' ? new Date(msg.timestamp).getTime() : msg.timestamp;
            if (firstTimestamp === null) {
                firstTimestamp = timestamp;
            }
            lastTimestamp = timestamp;

            // 如果是ASSISTANT消息，检查下一个消息是否是USER消息
            if (msg.role === 'ASSISTANT' && index < originalMemory.length - 1) {
                const nextMsg = originalMemory[index + 1];
                if (nextMsg && nextMsg.role === 'USER' && nextMsg.timestamp) {
                    const nextTimestamp = typeof nextMsg.timestamp === 'string' ? new Date(nextMsg.timestamp).getTime() : nextMsg.timestamp;
                    const gap = nextTimestamp - timestamp;
                    if (gap > 0) {
                        agentReplyToUserGap += gap;
                    }
                }
            }
        }
    });

    // 计算会话总时长（总时长减去Agent回复到用户再次回复的时间差）
    if (firstTimestamp !== null && lastTimestamp !== null) {
        const totalTime = lastTimestamp - firstTimestamp;
        sessionDuration = Math.max(0, totalTime - agentReplyToUserGap) / 1000; // 转换为秒
    }

    // 显示会话总览
    if (overviewDiv && (tokenCount > 0 || originalMemory.length > 0)) {
        overviewDiv.style.display = 'block';
        document.getElementById('originalTotalTokens').textContent = totalTokens.toLocaleString();
        document.getElementById('originalAvgTokens').textContent = tokenCount > 0 ? Math.round(totalTokens / tokenCount).toLocaleString() : '0';
        document.getElementById('originalMaxTokens').textContent = maxTokens.toLocaleString();
        document.getElementById('originalTabTotalMessages').textContent = originalMemory.length.toLocaleString();
        // 格式化推理总耗时为"xx小时xx分xx秒"
        const inferenceHours = Math.floor(totalInferenceTime / 3600);
        const inferenceMinutes = Math.floor((totalInferenceTime % 3600) / 60);
        const inferenceSeconds = Math.floor(totalInferenceTime % 60);
        let inferenceTimeText = '';
        if (inferenceHours > 0) {
            inferenceTimeText += `${inferenceHours}小时`;
        }
        if (inferenceMinutes > 0 || inferenceHours > 0) {
            inferenceTimeText += `${inferenceMinutes}分`;
        }
        inferenceTimeText += `${inferenceSeconds}秒`;
        document.getElementById('originalTotalInferenceTime').textContent = inferenceTimeText;
        // 格式化会话总时长为"xx小时xx分xx秒"
        const hours = Math.floor(sessionDuration / 3600);
        const minutes = Math.floor((sessionDuration % 3600) / 60);
        const seconds = Math.floor(sessionDuration % 60);
        let durationText = '';
        if (hours > 0) {
            durationText += `${hours}小时`;
        }
        if (minutes > 0 || hours > 0) {
            durationText += `${minutes}分`;
        }
        durationText += `${seconds}秒`;
        document.getElementById('originalSessionDuration').textContent = durationText;

        // 保存数据用于重绘
        originalTokenChartData = { tokenData, totalTokens, maxTokens };
        originalTimeChartData = { timeData, totalTime: totalInferenceTime, maxTime };
        
        // 绘制趋势图表
        drawTokenChart(tokenData, totalTokens, maxTokens);
        
        // 绘制推理耗时趋势图表
        drawTimeChart(timeData, totalInferenceTime, maxTime);
        
        // 渲染Top 10 Token消耗消息预览
        renderTop10TokenPreview(data, tokenRanking);
        
        // 监听窗口大小变化，重新绘制图表
        if (!window.originalTokenChartResizeHandler) {
            window.originalTokenChartResizeHandler = function() {
                if (originalTokenChartData) {
                    setTimeout(() => {
                        drawTokenChart(
                            originalTokenChartData.tokenData,
                            originalTokenChartData.totalTokens,
                            originalTokenChartData.maxTokens
                        );
                    }, 100);
                }
                if (originalTimeChartData) {
                    setTimeout(() => {
                        drawTimeChart(
                            originalTimeChartData.timeData,
                            originalTimeChartData.totalTime,
                            originalTimeChartData.maxTime
                        );
                    }, 100);
                }
            };
            window.addEventListener('resize', window.originalTokenChartResizeHandler);
        }
    } else if (overviewDiv) {
        overviewDiv.style.display = 'none';
        originalTokenChartData = null;
        originalTimeChartData = null;
    }

    // 渲染消息列表
    container.innerHTML = '';
    originalMemory.forEach((msg, index) => {
        const messageDiv = createMessageElement(msg, index, data, tokenRanking);
        container.appendChild(messageDiv);
    });
}

// 绘制Token消耗趋势图表
function drawTokenChart(tokenData, totalTokens, maxTokens) {
    const canvas = document.getElementById('originalTokenChart');
    if (!canvas) return;

    const ctx = canvas.getContext('2d');
    const width = canvas.offsetWidth || 800;
    const height = 300;
    canvas.width = width;
    canvas.height = height;

    // 清除画布
    ctx.clearRect(0, 0, width, height);

    if (tokenData.length === 0) {
        ctx.fillStyle = '#999';
        ctx.font = '14px Arial';
        ctx.textAlign = 'center';
        ctx.fillText('无Token数据', width / 2, height / 2);
        return;
    }

    // 设置边距
    const padding = { top: 40, right: 40, bottom: 60, left: 60 };
    const chartWidth = width - padding.left - padding.right;
    const chartHeight = height - padding.top - padding.bottom;

    // 计算数据范围
    const maxValue = Math.max(maxTokens, totalTokens / tokenData.length * 2) || 1000;
    const stepX = chartWidth / Math.max(tokenData.length - 1, 1);
    const scaleY = chartHeight / maxValue;

    // 绘制网格线
    ctx.strokeStyle = '#e0e0e0';
    ctx.lineWidth = 1;
    const gridLines = 5;
    for (let i = 0; i <= gridLines; i++) {
        const y = padding.top + (chartHeight / gridLines) * i;
        ctx.beginPath();
        ctx.moveTo(padding.left, y);
        ctx.lineTo(width - padding.right, y);
        ctx.stroke();
    }

    // 绘制坐标轴
    ctx.strokeStyle = '#333';
    ctx.lineWidth = 2;
    // Y轴
    ctx.beginPath();
    ctx.moveTo(padding.left, padding.top);
    ctx.lineTo(padding.left, height - padding.bottom);
    ctx.stroke();
    // X轴
    ctx.beginPath();
    ctx.moveTo(padding.left, height - padding.bottom);
    ctx.lineTo(width - padding.right, height - padding.bottom);
    ctx.stroke();

    // 绘制Y轴标签
    ctx.fillStyle = '#666';
    ctx.font = '12px Arial';
    ctx.textAlign = 'right';
    ctx.textBaseline = 'middle';
    for (let i = 0; i <= gridLines; i++) {
        const value = maxValue - (maxValue / gridLines) * i;
        const y = padding.top + (chartHeight / gridLines) * i;
        ctx.fillText(Math.round(value).toLocaleString(), padding.left - 10, y);
    }

    // 绘制X轴标签（每10个点显示一个，或最多显示10个）
    ctx.textAlign = 'center';
    ctx.textBaseline = 'top';
    const labelStep = Math.max(1, Math.floor(tokenData.length / 10));
    for (let i = 0; i < tokenData.length; i += labelStep) {
        const x = padding.left + stepX * i;
        if (x <= width - padding.right) {
            ctx.fillText(`#${i + 1}`, x, height - padding.bottom + 10);
        }
    }

    // 绘制折线图
    ctx.strokeStyle = '#28a745';
    ctx.lineWidth = 2;
    ctx.beginPath();
    
    let hasData = false;
    for (let i = 0; i < tokenData.length; i++) {
        const x = padding.left + stepX * i;
        const y = height - padding.bottom - (tokenData[i].tokens * scaleY);
        
        if (i === 0) {
            ctx.moveTo(x, y);
        } else {
            ctx.lineTo(x, y);
        }
        
        if (tokenData[i].tokens > 0) {
            hasData = true;
        }
    }
    
    if (hasData) {
        ctx.stroke();
        
        // 绘制数据点
        ctx.fillStyle = '#28a745';
        for (let i = 0; i < tokenData.length; i++) {
            if (tokenData[i].tokens > 0) {
                const x = padding.left + stepX * i;
                const y = height - padding.bottom - (tokenData[i].tokens * scaleY);
                ctx.beginPath();
                ctx.arc(x, y, 3, 0, Math.PI * 2);
                ctx.fill();
            }
        }
    } else {
        // 如果没有数据，显示提示
        ctx.fillStyle = '#999';
        ctx.font = '14px Arial';
        ctx.textAlign = 'center';
        ctx.fillText('无Token数据', width / 2, height / 2);
    }

    // 绘制标题
    ctx.fillStyle = '#333';
    ctx.font = 'bold 14px Arial';
    ctx.textAlign = 'center';
    ctx.fillText('Token消耗趋势（按消息顺序）', width / 2, 20);
}

// 绘制推理耗时趋势图表
function drawTimeChart(timeData, totalTime, maxTime) {
    const canvas = document.getElementById('originalTimeChart');
    if (!canvas) return;

    const ctx = canvas.getContext('2d');
    const width = canvas.offsetWidth || 800;
    const height = 300;
    canvas.width = width;
    canvas.height = height;

    // 清除画布
    ctx.clearRect(0, 0, width, height);

    if (timeData.length === 0) {
        ctx.fillStyle = '#999';
        ctx.font = '14px Arial';
        ctx.textAlign = 'center';
        ctx.fillText('无推理耗时数据', width / 2, height / 2);
        return;
    }

    // 设置边距
    const padding = { top: 40, right: 40, bottom: 60, left: 60 };
    const chartWidth = width - padding.left - padding.right;
    const chartHeight = height - padding.top - padding.bottom;

    // 计算数据范围
    const maxValue = Math.max(maxTime, totalTime / timeData.length * 2) || 10;
    const stepX = chartWidth / Math.max(timeData.length - 1, 1);
    const scaleY = chartHeight / maxValue;

    // 绘制网格线
    ctx.strokeStyle = '#e0e0e0';
    ctx.lineWidth = 1;
    const gridLines = 5;
    for (let i = 0; i <= gridLines; i++) {
        const y = padding.top + (chartHeight / gridLines) * i;
        ctx.beginPath();
        ctx.moveTo(padding.left, y);
        ctx.lineTo(width - padding.right, y);
        ctx.stroke();
    }

    // 绘制坐标轴
    ctx.strokeStyle = '#333';
    ctx.lineWidth = 2;
    // Y轴
    ctx.beginPath();
    ctx.moveTo(padding.left, padding.top);
    ctx.lineTo(padding.left, height - padding.bottom);
    ctx.stroke();
    // X轴
    ctx.beginPath();
    ctx.moveTo(padding.left, height - padding.bottom);
    ctx.lineTo(width - padding.right, height - padding.bottom);
    ctx.stroke();

    // 绘制Y轴标签（显示秒数，保留1位小数）
    ctx.fillStyle = '#666';
    ctx.font = '12px Arial';
    ctx.textAlign = 'right';
    ctx.textBaseline = 'middle';
    for (let i = 0; i <= gridLines; i++) {
        const value = maxValue - (maxValue / gridLines) * i;
        const y = padding.top + (chartHeight / gridLines) * i;
        ctx.fillText(value.toFixed(1) + 's', padding.left - 10, y);
    }

    // 绘制X轴标签（每10个点显示一个，或最多显示10个）
    ctx.textAlign = 'center';
    ctx.textBaseline = 'top';
    const labelStep = Math.max(1, Math.floor(timeData.length / 10));
    for (let i = 0; i < timeData.length; i += labelStep) {
        const x = padding.left + stepX * i;
        if (x <= width - padding.right) {
            ctx.fillText(`#${i + 1}`, x, height - padding.bottom + 10);
        }
    }

    // 绘制折线图
    ctx.strokeStyle = '#ff9800';
    ctx.lineWidth = 2;
    ctx.beginPath();
    
    let hasData = false;
    for (let i = 0; i < timeData.length; i++) {
        const x = padding.left + stepX * i;
        const y = height - padding.bottom - (timeData[i].time * scaleY);
        
        if (i === 0) {
            ctx.moveTo(x, y);
        } else {
            ctx.lineTo(x, y);
        }
        
        if (timeData[i].time > 0) {
            hasData = true;
        }
    }
    
    if (hasData) {
        ctx.stroke();
        
        // 绘制数据点
        ctx.fillStyle = '#ff9800';
        for (let i = 0; i < timeData.length; i++) {
            if (timeData[i].time > 0) {
                const x = padding.left + stepX * i;
                const y = height - padding.bottom - (timeData[i].time * scaleY);
                ctx.beginPath();
                ctx.arc(x, y, 3, 0, Math.PI * 2);
                ctx.fill();
            }
        }
    } else {
        // 如果没有数据，显示提示
        ctx.fillStyle = '#999';
        ctx.font = '14px Arial';
        ctx.textAlign = 'center';
        ctx.fillText('无推理耗时数据', width / 2, height / 2);
    }

    // 绘制标题
    ctx.fillStyle = '#333';
    ctx.font = 'bold 14px Arial';
    ctx.textAlign = 'center';
    ctx.fillText('推理耗时趋势（按消息顺序）', width / 2, 20);
}

// 渲染Top 10 Token消耗消息预览
function renderTop10TokenPreview(data, tokenRanking) {
    const container = document.getElementById('originalTop10Preview');
    if (!container) return;
    
    container.innerHTML = '';
    
    // 获取Top 10 token消耗的消息
    const topMessages = getTopTokenUsageMessages(data, 10);
    
    if (!topMessages || topMessages.length === 0) {
        container.innerHTML = '<div style="color: #999; font-size: 12px; text-align: center; padding: 20px;">未找到Token使用数据</div>';
        return;
    }
    
    topMessages.forEach((item, index) => {
        const { message, totalTokens, rank } = item;
        
        // 创建预览卡片
        const previewCard = document.createElement('div');
        previewCard.style.background = '#f8f9fa';
        previewCard.style.border = '1px solid #e0e0e0';
        previewCard.style.borderRadius = '6px';
        previewCard.style.padding = '8px';
        previewCard.style.marginBottom = '8px';
        previewCard.style.fontSize = '11px';
        
        // 排名和Token信息
        const headerDiv = document.createElement('div');
        headerDiv.style.display = 'flex';
        headerDiv.style.justifyContent = 'space-between';
        headerDiv.style.alignItems = 'center';
        headerDiv.style.marginBottom = '6px';
        
        const rankBadge = document.createElement('span');
        rankBadge.style.background = rank === 1 ? '#dc3545' : (rank <= 3 ? '#ff9800' : '#6c757d');
        rankBadge.style.color = '#fff';
        rankBadge.style.padding = '2px 6px';
        rankBadge.style.borderRadius = '8px';
        rankBadge.style.fontSize = '10px';
        rankBadge.style.fontWeight = 'bold';
        rankBadge.textContent = `#${rank}`;
        headerDiv.appendChild(rankBadge);
        
        const tokenInfo = document.createElement('span');
        tokenInfo.style.fontSize = '10px';
        tokenInfo.style.color = '#666';
        tokenInfo.style.fontWeight = 'bold';
        tokenInfo.textContent = `${totalTokens.toLocaleString()}`;
        headerDiv.appendChild(tokenInfo);
        
        previewCard.appendChild(headerDiv);
        
        // 消息预览文本（只显示前80个字符）
        const previewText = document.createElement('div');
        previewText.style.color = '#333';
        previewText.style.fontSize = '10px';
        previewText.style.lineHeight = '1.4';
        previewText.style.maxHeight = '40px';
        previewText.style.overflow = 'hidden';
        previewText.style.textOverflow = 'ellipsis';
        previewText.style.display = '-webkit-box';
        previewText.style.webkitLineClamp = '2';
        previewText.style.webkitBoxOrient = 'vertical';
        
        const fullPreview = getFullMessagePreview(message);
        if (fullPreview.length > 80) {
            previewText.textContent = fullPreview.substring(0, 80) + '...';
        } else {
            previewText.textContent = fullPreview;
        }
        
        previewCard.appendChild(previewText);
        
        // 添加定位按钮
        const locateBtn = document.createElement('span');
        locateBtn.innerHTML = '🔗';
        locateBtn.style.cursor = 'pointer';
        locateBtn.style.float = 'right';
        locateBtn.style.marginTop = '4px';
        locateBtn.style.fontSize = '10px';
        locateBtn.style.opacity = '0.7';
        locateBtn.style.transition = 'opacity 0.2s';
        locateBtn.title = '定位到消息';
        locateBtn.onmouseenter = function() {
            this.style.opacity = '1';
        };
        locateBtn.onmouseleave = function() {
            this.style.opacity = '0.7';
        };
        locateBtn.onclick = function(e) {
            e.stopPropagation();
            e.preventDefault();
            
            // 生成与createMessageElement中完全相同的消息ID
            let messageId = message.id;
            if (!messageId) {
                const role = message.role || 'unknown';
                const name = message.name || 'unknown';
                const timestamp = message.timestamp || '';
                const contentHash = message.content ? JSON.stringify(message.content).substring(0, 100) : '';
                messageId = `msg_${role}_${name}_${timestamp}_${contentHash}`.replace(/[^a-zA-Z0-9_]/g, '_');
            }
            const targetId = `message-${messageId}`;
            
            // 定位函数
            function findAndScroll() {
                let targetElement = document.getElementById(targetId);
                
                // 如果通过ID找不到，尝试通过内容匹配
                if (!targetElement) {
                    const allMessages = document.querySelectorAll('.message');
                    const messagePreview = getFullMessagePreview(message);
                    const previewStart = messagePreview.substring(0, 100);
                    
                    for (let msgEl of allMessages) {
                        const previewDiv = msgEl.querySelector('.message-preview');
                        if (previewDiv) {
                            const previewText = previewDiv.textContent || '';
                            if (previewText && previewStart && 
                                (previewText.includes(previewStart.substring(0, 50)) || 
                                 previewStart.substring(0, 50).includes(previewText.substring(0, 50)))) {
                                targetElement = msgEl;
                                break;
                            }
                        }
                    }
                }
                
                if (targetElement) {
                    // 确保消息展开
                    if (targetElement.classList.contains('collapsed')) {
                        targetElement.classList.remove('collapsed');
                    }
                    
                    // 滚动到目标元素
                    targetElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
                    
                    // 高亮显示
                    targetElement.style.transition = 'background-color 0.3s, box-shadow 0.3s';
                    const originalBg = targetElement.style.backgroundColor || '';
                    const originalShadow = targetElement.style.boxShadow || '';
                    targetElement.style.backgroundColor = '#fff3cd';
                    targetElement.style.boxShadow = '0 0 10px rgba(255, 193, 7, 0.5)';
                    
                    setTimeout(() => {
                        targetElement.style.backgroundColor = originalBg;
                        targetElement.style.boxShadow = originalShadow;
                        setTimeout(() => {
                            targetElement.style.transition = '';
                        }, 300);
                    }, 2000);
                } else {
                    alert('无法定位到该消息，可能该消息不在当前显示的内存列表中');
                }
            }
            
            // 先尝试在当前tab中查找
            let targetElement = document.getElementById(targetId);
            
            // 如果找不到，切换到原始内存tab
            if (!targetElement) {
                const originalTab = document.querySelector('[data-tab="original"]');
                if (originalTab && !originalTab.classList.contains('active')) {
                    originalTab.click();
                }
                setTimeout(findAndScroll, 300);
            } else {
                findAndScroll();
            }
        };
        
        previewCard.appendChild(locateBtn);
        container.appendChild(previewCard);
    });
}

