// working-memory.js
// 依赖: common.js, message-utils.js

function renderWorkingMemory(data, tokenRanking) {
    // 提取workingMemoryStorage
    let workingMemory = [];
    
    if (data.agent && data.agent.memory && data.agent.memory.workingMemoryStorage) {
        workingMemory = data.agent.memory.workingMemoryStorage;
    } else if (data.workingMemoryStorage) {
        workingMemory = data.workingMemoryStorage;
    } else if (Array.isArray(data)) {
        workingMemory = data;
    }

    // 提取originalMemoryStorage
    let originalMemory = [];
    if (data.agent && data.agent.memory && data.agent.memory.originalMemoryStorage) {
        originalMemory = data.agent.memory.originalMemoryStorage;
    } else if (data.originalMemoryStorage) {
        originalMemory = data.originalMemoryStorage;
    }

    if (!workingMemory || workingMemory.length === 0) {
        document.getElementById('workingMessagesContainer').innerHTML = 
            '<div class="empty-state"><div class="empty-state-icon">⚠️</div><div>未找到workingMemoryStorage数据</div></div>';
        return;
    }

    // 统计信息
    const stats = {
        total: workingMemory.length,
        user: 0,
        assistant: 0,
        tool: 0,
        memory: 0
    };

    workingMemory.forEach(msg => {
        if (msg.role === 'USER') stats.user++;
        else if (msg.role === 'ASSISTANT') stats.assistant++;
        else if (msg.role === 'TOOL') stats.tool++;
        if (msg.name === 'long_term_memory') stats.memory++;
    });

    // 计算工作内存总字符数
    const workingTotalChars = calculateTextLength(workingMemory);
    
    // 计算对话轮次（USER消息数）
    const workingRounds = stats.user;
    
    // 区域一和区域二：工作内存总揽和原始内存总揽
    document.getElementById('memoryOverview').style.display = 'flex';
    document.getElementById('workingTotalMessages').textContent = stats.total;
    document.getElementById('workingTotalChars').textContent = workingTotalChars.toLocaleString();
    document.getElementById('workingRounds').textContent = workingRounds;
    document.getElementById('workingToolMessages').textContent = stats.tool;

    // 获取offloadContext
    const offloadContext = getOffloadContext(data);
    
    // 计算卸载区内存块数和总消息数
    let offloadBlocks = 0;
    let offloadTotalMessages = 0;
    if (offloadContext) {
        offloadBlocks = Object.keys(offloadContext).length;
        Object.values(offloadContext).forEach(block => {
            if (Array.isArray(block)) {
                offloadTotalMessages += block.length;
            }
        });
    }
    
    // 计算原始内存总字符数
    const originalTotalChars = originalMemory && originalMemory.length > 0 
        ? calculateTextLength(originalMemory) 
        : 0;
    
    // 计算下降百分比
    const originalCount = originalMemory && originalMemory.length > 0 ? originalMemory.length : 0;
    const messageDecline = originalCount > 0 && stats.total < originalCount
        ? `(下降 ${((1 - stats.total / originalCount) * 100).toFixed(1)}%)`
        : '';
    const charDecline = originalTotalChars > 0 && workingTotalChars < originalTotalChars
        ? `(下降 ${((1 - workingTotalChars / originalTotalChars) * 100).toFixed(1)}%)`
        : '';
    
    // 区域一：合并后的内存总览（已在同一行中）
    document.getElementById('originalTotalMessages').textContent = originalCount || 0;
    document.getElementById('messageDecline').textContent = messageDecline;
    document.getElementById('originalTotalChars').textContent = originalTotalChars.toLocaleString();
    document.getElementById('charDecline').textContent = charDecline;
    document.getElementById('offloadBlocks').textContent = offloadBlocks;
    document.getElementById('offloadTotalMessages').textContent = offloadTotalMessages;

    // 区域二：渲染Token消耗 Top 5 预览
    const top5Container = document.getElementById('top5TokenPreview');
    top5Container.innerHTML = '';
    
    const topMessages = getTopTokenUsageMessages(data, 5);
    
    if (!topMessages || topMessages.length === 0) {
        top5Container.innerHTML = '<div style="color: #999; font-size: 12px; text-align: center; padding: 20px;">未找到Token使用数据</div>';
    } else {
        topMessages.forEach((item, index) => {
            const { message, totalTokens, rank } = item;
            
            // 创建预览卡片
            const previewCard = document.createElement('div');
            previewCard.style.background = '#f8f9fa';
            previewCard.style.border = '1px solid #e0e0e0';
            previewCard.style.borderRadius = '6px';
            previewCard.style.padding = '10px';
            previewCard.style.marginBottom = '10px';
            previewCard.style.fontSize = '12px';
            
            // 排名和Token信息
            const headerDiv = document.createElement('div');
            headerDiv.style.display = 'flex';
            headerDiv.style.justifyContent = 'space-between';
            headerDiv.style.alignItems = 'center';
            headerDiv.style.marginBottom = '8px';
            
            const rankBadge = document.createElement('span');
            rankBadge.style.background = rank === 1 ? '#dc3545' : (rank <= 3 ? '#ff9800' : '#6c757d');
            rankBadge.style.color = '#fff';
            rankBadge.style.padding = '2px 8px';
            rankBadge.style.borderRadius = '10px';
            rankBadge.style.fontSize = '11px';
            rankBadge.style.fontWeight = 'bold';
            rankBadge.textContent = `#${rank}`;
            headerDiv.appendChild(rankBadge);
            
            const tokenInfo = document.createElement('span');
            tokenInfo.style.fontSize = '11px';
            tokenInfo.style.color = '#666';
            tokenInfo.style.fontWeight = 'bold';
            tokenInfo.textContent = `${totalTokens.toLocaleString()} tokens`;
            headerDiv.appendChild(tokenInfo);
            
            // 添加定位按钮
            const locateBtn = document.createElement('span');
            locateBtn.innerHTML = '🔗';
            locateBtn.style.cursor = 'pointer';
            locateBtn.style.marginLeft = '8px';
            locateBtn.style.fontSize = '12px';
            locateBtn.style.opacity = '0.7';
            locateBtn.style.transition = 'opacity 0.2s';
            locateBtn.title = '定位到消息';
            locateBtn.onmouseenter = function() {
                this.style.opacity = '1';
            };
            locateBtn.onmouseleave = function() {
                this.style.opacity = '0.7';
            };
            locateBtn.onclick = function(e) {
                e.stopPropagation();
                e.preventDefault();
                
                // 生成与createMessageElement中完全相同的消息ID
                let messageId = message.id;
                if (!messageId) {
                    const role = message.role || 'unknown';
                    const name = message.name || 'unknown';
                    const timestamp = message.timestamp || '';
                    const contentHash = message.content ? JSON.stringify(message.content).substring(0, 100) : '';
                    messageId = `msg_${role}_${name}_${timestamp}_${contentHash}`.replace(/[^a-zA-Z0-9_]/g, '_');
                }
                const targetId = `message-${messageId}`;
                
                console.log('🔗 定位消息:', {
                    messageId: messageId,
                    targetId: targetId,
                    message: message
                });
                
                // 定位函数
                function findAndScroll() {
                    let targetElement = document.getElementById(targetId);
                    console.log('通过ID查找结果:', !!targetElement, targetId);
                    
                    // 如果通过ID找不到，尝试通过内容匹配
                    if (!targetElement) {
                        const allMessages = document.querySelectorAll('.message');
                        console.log('找到的消息元素数量:', allMessages.length);
                        
                        const messagePreview = getFullMessagePreview(message);
                        const previewStart = messagePreview.substring(0, 100);
                        console.log('消息预览:', previewStart);
                        
                        for (let msgEl of allMessages) {
                            // 检查消息预览文本
                            const previewDiv = msgEl.querySelector('.message-preview');
                            if (previewDiv) {
                                const previewText = previewDiv.textContent || '';
                                if (previewText && previewStart && 
                                    (previewText.includes(previewStart.substring(0, 50)) || 
                                     previewStart.substring(0, 50).includes(previewText.substring(0, 50)))) {
                                    targetElement = msgEl;
                                    console.log('通过内容匹配找到:', msgEl.id);
                                    break;
                                }
                            }
                            
                            // 检查消息头部信息（role和name）
                            const header = msgEl.querySelector('.message-header');
                            if (header) {
                                const headerText = header.textContent || '';
                                const role = message.role || '';
                                const name = message.name || '';
                                if (role && headerText.includes(role) && 
                                    (!name || headerText.includes(name))) {
                                    // 进一步检查内容是否匹配
                                    const msgContent = msgEl.textContent || '';
                                    if (msgContent.includes(previewStart.substring(0, 30))) {
                                        targetElement = msgEl;
                                        console.log('通过头部和内容匹配找到:', msgEl.id);
                                        break;
                                    }
                                }
                            }
                        }
                    }
                    
                    if (targetElement) {
                        console.log('✅ 找到目标消息，开始滚动:', targetElement.id);
                        
                        // 确保消息展开（移除collapsed类）
                        if (targetElement.classList.contains('collapsed')) {
                            targetElement.classList.remove('collapsed');
                            console.log('展开消息');
                        }
                        
                        // 确保消息容器可见
                        const container = targetElement.closest('.messages-container') || 
                                         targetElement.closest('.tab-content');
                        if (container) {
                            const containerRect = container.getBoundingClientRect();
                            if (containerRect.height === 0 || containerRect.width === 0) {
                                console.log('容器不可见，尝试显示');
                                container.style.display = 'block';
                            }
                        }
                        
                        // 等待DOM更新
                        setTimeout(() => {
                            // 获取目标元素的位置
                            const rect = targetElement.getBoundingClientRect();
                            const elementTop = rect.top + window.pageYOffset;
                            const elementHeight = rect.height;
                            const windowHeight = window.innerHeight;
                            
                            // 计算滚动位置（居中显示）
                            const scrollPosition = elementTop - (windowHeight / 2) + (elementHeight / 2);
                            
                            console.log('滚动信息:', {
                                elementTop: elementTop,
                                elementHeight: elementHeight,
                                windowHeight: windowHeight,
                                scrollPosition: scrollPosition,
                                currentScrollY: window.pageYOffset,
                                elementRect: rect
                            });
                            
                            // 先尝试scrollIntoView（更可靠）
                            try {
                                targetElement.scrollIntoView({ 
                                    behavior: 'smooth', 
                                    block: 'center',
                                    inline: 'nearest'
                                });
                                
                                // 如果scrollIntoView不起作用，使用window.scrollTo作为备用
                                setTimeout(() => {
                                    const newRect = targetElement.getBoundingClientRect();
                                    const newTop = newRect.top + window.pageYOffset;
                                    const newScrollPos = newTop - (window.innerHeight / 2) + (newRect.height / 2);
                                    
                                    // 检查是否已经滚动到目标位置（允许50px误差）
                                    if (Math.abs(window.pageYOffset - newScrollPos) > 50) {
                                        console.log('scrollIntoView未生效，使用window.scrollTo');
                                        window.scrollTo({
                                            top: newScrollPos,
                                            behavior: 'smooth'
                                        });
                                    } else {
                                        console.log('滚动成功，当前位置:', window.pageYOffset);
                                    }
                                }, 300);
                            } catch (e) {
                                console.error('scrollIntoView失败:', e);
                                // 降级到window.scrollTo
                                window.scrollTo({
                                    top: scrollPosition,
                                    behavior: 'smooth'
                                });
                            }
                            
                            // 高亮显示
                            targetElement.style.transition = 'background-color 0.3s, box-shadow 0.3s';
                            const originalBg = targetElement.style.backgroundColor || '';
                            const originalShadow = targetElement.style.boxShadow || '';
                            targetElement.style.backgroundColor = '#fff3cd';
                            targetElement.style.boxShadow = '0 0 10px rgba(255, 193, 7, 0.5)';
                            
                            setTimeout(() => {
                                targetElement.style.backgroundColor = originalBg;
                                targetElement.style.boxShadow = originalShadow;
                                setTimeout(() => {
                                    targetElement.style.transition = '';
                                }, 300);
                            }, 2000);
                        }, 50);
                    } else {
                        console.error('❌ 无法找到消息:', targetId);
                        // 列出所有消息ID以便调试
                        const allMsgIds = Array.from(document.querySelectorAll('.message')).map(el => el.id);
                        console.log('所有消息ID:', allMsgIds);
                        alert('无法定位到该消息，可能该消息不在当前显示的内存列表中。请打开浏览器控制台查看详细信息。');
                    }
                }
                
                // 先尝试在当前tab中查找
                let targetElement = document.getElementById(targetId);
                
                if (targetElement) {
                    findAndScroll();
                } else {
                    // 如果找不到，尝试切换到工作内存tab
                    const workingTab = document.querySelector('[data-tab="working"]');
                    if (workingTab && !workingTab.classList.contains('active')) {
                        console.log('切换到工作内存tab');
                        workingTab.click();
                    }
                    // 等待DOM更新后查找
                    setTimeout(findAndScroll, 300);
                    
                    // 如果还是找不到，尝试切换到原始内存tab
                    setTimeout(() => {
                        if (!document.getElementById(targetId)) {
                            const originalTab = document.querySelector('[data-tab="original"]');
                            if (originalTab && !originalTab.classList.contains('active')) {
                                console.log('切换到原始内存tab');
                                originalTab.click();
                                setTimeout(findAndScroll, 300);
                            }
                        }
                    }, 600);
                }
            };
            headerDiv.appendChild(locateBtn);
            
            previewCard.appendChild(headerDiv);
            
            // 消息预览文本（只显示前100个字符）
            const previewText = document.createElement('div');
            previewText.style.color = '#333';
            previewText.style.fontSize = '11px';
            previewText.style.lineHeight = '1.4';
            previewText.style.maxHeight = '60px';
            previewText.style.overflow = 'hidden';
            previewText.style.textOverflow = 'ellipsis';
            
            const fullPreview = getFullMessagePreview(message);
            if (fullPreview.length > 100) {
                previewText.textContent = fullPreview.substring(0, 100) + '...';
            } else {
                previewText.textContent = fullPreview;
            }
            
            previewCard.appendChild(previewText);
            top5Container.appendChild(previewCard);
        });
    }

    // 渲染工作内存消息
    const container = document.getElementById('workingMessagesContainer');
    container.innerHTML = '';

    workingMemory.forEach((msg, index) => {
        const messageDiv = createMessageElement(msg, index, data, tokenRanking);
        container.appendChild(messageDiv);
    });
}

